/*
 * Decompiled with CFR 0.152.
 */
package com.skynex.mylands.service;

import com.skynex.mylands.database.LandRepository;
import com.skynex.mylands.libs.caffeine.cache.Cache;
import com.skynex.mylands.libs.caffeine.cache.Caffeine;
import com.skynex.mylands.model.Cuboid;
import com.skynex.mylands.model.Land;
import com.skynex.mylands.model.LandLevel;
import com.skynex.mylands.util.PluginLogger;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.bukkit.Location;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class LandService {
    private final LandRepository repository;
    private final Cache<UUID, Land> landCache;
    private final Map<UUID, Set<UUID>> pendingInvitations;
    private final int initialSize;
    private final int heightLimit;
    private final int depthLimit;
    private final int maxFarmSize;
    private final int sizeIncreasePerLevel;
    private final int maxLevel;
    private final int maxCoOwners;
    private final Cache<UUID, UUID> coOwnerToOwnerCache;

    public LandService(@NotNull LandRepository repository, int initialSize, int heightLimit, int depthLimit, int maxFarmSize, int sizeIncreasePerLevel, int maxLevel, int maxCoOwners) {
        this.repository = repository;
        this.initialSize = initialSize;
        this.heightLimit = heightLimit;
        this.depthLimit = depthLimit;
        this.maxFarmSize = maxFarmSize;
        this.sizeIncreasePerLevel = sizeIncreasePerLevel;
        this.maxLevel = maxLevel;
        this.maxCoOwners = maxCoOwners;
        this.landCache = Caffeine.newBuilder().expireAfterWrite(30L, TimeUnit.MINUTES).expireAfterAccess(15L, TimeUnit.MINUTES).maximumSize(1000L).build();
        this.coOwnerToOwnerCache = Caffeine.newBuilder().expireAfterWrite(15L, TimeUnit.MINUTES).maximumSize(500L).build();
        this.pendingInvitations = new ConcurrentHashMap<UUID, Set<UUID>>();
    }

    public CompletableFuture<Void> initialize() {
        return this.repository.loadAllLands().thenAccept(lands -> {
            lands.forEach(land -> this.landCache.put(land.ownerId(), (Land)land));
            PluginLogger.info("Loaded {} lands into cache", lands.size());
        });
    }

    public CompletableFuture<Land> createLand(@NotNull UUID ownerId, @NotNull String name, @NotNull Location location) {
        return CompletableFuture.supplyAsync(() -> {
            if (this.landCache.getIfPresent(ownerId) != null) {
                throw new IllegalStateException("Player already has a land");
            }
            Cuboid cuboid = this.createCuboidForFarm(location, this.initialSize);
            for (Land existingFarm : this.landCache.asMap().values()) {
                if (!cuboid.overlaps(existingFarm.cuboid())) continue;
                throw new IllegalStateException("This area overlaps with another land");
            }
            Location defaultHome = this.calculateDefaultHome(cuboid);
            Land land = new Land(ownerId, name, cuboid, 1, new HashSet<UUID>(), defaultHome, true);
            this.repository.saveLand(land).join();
            this.landCache.put(ownerId, land);
            PluginLogger.info("Created land '{}' for player {}", name, ownerId);
            return land;
        });
    }

    public CompletableFuture<Boolean> deleteLand(@NotNull UUID ownerId) {
        return this.repository.deleteLand(ownerId).thenApply(success -> {
            if (success.booleanValue()) {
                this.landCache.invalidate(ownerId);
                this.pendingInvitations.remove(ownerId);
                PluginLogger.info("Deleted land for player {}", ownerId);
            }
            return success;
        });
    }

    public CompletableFuture<Optional<Land>> getLand(@NotNull UUID ownerId) {
        Land cached = this.landCache.getIfPresent(ownerId);
        if (cached != null) {
            return CompletableFuture.completedFuture(Optional.of(cached));
        }
        return this.repository.loadLand(ownerId).thenApply(landOpt -> {
            landOpt.ifPresent(land -> this.landCache.put(ownerId, (Land)land));
            return landOpt;
        });
    }

    public CompletableFuture<Optional<Land>> getLandForPlayer(@NotNull UUID playerId) {
        return this.getLand(playerId).thenCompose(landOpt -> {
            if (landOpt.isPresent()) {
                return CompletableFuture.completedFuture(landOpt);
            }
            UUID cachedOwnerId = this.coOwnerToOwnerCache.getIfPresent(playerId);
            if (cachedOwnerId != null) {
                return this.getLand(cachedOwnerId);
            }
            return CompletableFuture.supplyAsync(() -> this.landCache.asMap().values().stream().filter(land -> land.isCoOwner(playerId)).findFirst()).thenApply(foundLand -> {
                foundLand.ifPresent(land -> this.coOwnerToOwnerCache.put(playerId, land.ownerId()));
                return foundLand;
            });
        });
    }

    @Nullable
    public Land getLandAtLocation(@NotNull Location location) {
        return this.landCache.asMap().values().stream().filter(land -> land.contains(location)).findFirst().orElse(null);
    }

    public boolean hasAccessToLocation(@NotNull UUID playerId, @NotNull Location location) {
        Land land = this.getLandAtLocation(location);
        return land != null && land.hasAccess(playerId);
    }

    public CompletableFuture<Land> upgradeLand(@NotNull UUID ownerId, @NotNull LandLevel nextLevel) {
        return this.getLand(ownerId).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                throw new IllegalStateException("Land not found");
            }
            Land land = (Land)landOpt.get();
            if (land.level() >= this.maxLevel) {
                throw new IllegalStateException("Land is already at max level");
            }
            int newSize = this.calculateLandSize(nextLevel.level());
            Cuboid newCuboid = this.expandCuboidCentered(land.cuboid(), newSize);
            for (Land otherLand : this.landCache.asMap().values()) {
                if (otherLand.ownerId().equals(ownerId) || !newCuboid.overlaps(otherLand.cuboid())) continue;
                throw new IllegalStateException("Upgrade would overlap with another land");
            }
            Land upgradedFarm = land.withLevel(nextLevel.level()).withCuboid(newCuboid);
            return this.repository.saveLand(upgradedFarm).thenApply(v -> {
                this.landCache.put(ownerId, upgradedFarm);
                PluginLogger.info("Upgraded land for player {} to level {} (size: {}x{})", ownerId, nextLevel.level(), newSize, newSize);
                return upgradedFarm;
            });
        });
    }

    private Cuboid expandCuboidCentered(@NotNull Cuboid oldCuboid, int newSize) {
        Location center = oldCuboid.getCenter();
        double halfSize = (double)newSize / 2.0;
        Location corner1 = center.clone();
        corner1.add(-halfSize, 0.0, -halfSize);
        corner1.setY((double)this.depthLimit);
        Location corner2 = center.clone();
        corner2.add(halfSize, 0.0, halfSize);
        corner2.setY((double)this.heightLimit);
        return Cuboid.from(corner1, corner2);
    }

    public CompletableFuture<Land> setHome(@NotNull UUID ownerId, @NotNull Location home) {
        return this.getLand(ownerId).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                throw new IllegalStateException("Land not found");
            }
            Land land = (Land)landOpt.get();
            if (!land.contains(home)) {
                throw new IllegalArgumentException("Home must be within land boundaries");
            }
            Land updatedLand = land.withHome(home);
            return this.repository.updateHome(ownerId, home).thenApply(v -> {
                this.landCache.put(ownerId, updatedLand);
                return updatedLand;
            });
        });
    }

    public CompletableFuture<Land> setVisitsOpen(@NotNull UUID ownerId, boolean open) {
        return this.getLand(ownerId).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                throw new IllegalStateException("Land not found");
            }
            Land land = (Land)landOpt.get();
            Land updatedLand = land.withVisitsOpen(open);
            return this.repository.updateVisitsOpen(ownerId, open).thenApply(v -> {
                this.landCache.put(ownerId, updatedLand);
                return updatedLand;
            });
        });
    }

    public boolean inviteCoOwner(@NotNull UUID ownerId, @NotNull UUID targetId) {
        Land land = this.landCache.getIfPresent(ownerId);
        if (land == null) {
            return false;
        }
        if (land.coOwners().size() >= this.maxCoOwners) {
            throw new IllegalStateException("Maximum co-owners reached");
        }
        if (land.isCoOwner(targetId)) {
            throw new IllegalStateException("Player is already a co-owner");
        }
        this.pendingInvitations.computeIfAbsent(ownerId, k -> new HashSet()).add(targetId);
        return true;
    }

    public CompletableFuture<Land> acceptInvitation(@NotNull UUID playerId) {
        UUID ownerId = this.pendingInvitations.entrySet().stream().filter(entry -> ((Set)entry.getValue()).contains(playerId)).map(Map.Entry::getKey).findFirst().orElseThrow(() -> new IllegalStateException("No pending invitation found"));
        return this.getLand(ownerId).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                throw new IllegalStateException("Land not found");
            }
            Land land = (Land)landOpt.get();
            HashSet<UUID> newCoOwners = new HashSet<UUID>(land.coOwners());
            newCoOwners.add(playerId);
            Land updatedLand = land.withCoOwners(newCoOwners);
            return this.repository.updateCoOwners(ownerId, newCoOwners).thenApply(v -> {
                this.landCache.put(ownerId, updatedLand);
                this.coOwnerToOwnerCache.invalidateAll();
                this.pendingInvitations.get(ownerId).remove(playerId);
                PluginLogger.info("Player {} joined land owned by {}", playerId, ownerId);
                return updatedLand;
            });
        });
    }

    public CompletableFuture<Land> removeCoOwner(@NotNull UUID ownerId, @NotNull UUID coOwnerId) {
        return this.getLand(ownerId).thenCompose(landOpt -> {
            if (landOpt.isEmpty()) {
                throw new IllegalStateException("Land not found");
            }
            Land land = (Land)landOpt.get();
            if (!land.isCoOwner(coOwnerId)) {
                throw new IllegalArgumentException("Player is not a co-owner");
            }
            HashSet<UUID> newCoOwners = new HashSet<UUID>(land.coOwners());
            newCoOwners.remove(coOwnerId);
            Land updatedLand = land.withCoOwners(newCoOwners);
            return this.repository.updateCoOwners(ownerId, newCoOwners).thenApply(v -> {
                this.landCache.put(ownerId, updatedLand);
                this.coOwnerToOwnerCache.invalidateAll();
                ((Set)this.pendingInvitations.getOrDefault(ownerId, new HashSet())).remove(coOwnerId);
                return updatedLand;
            });
        });
    }

    public CompletableFuture<Void> quitLand(@NotNull UUID playerId) {
        UUID ownerId = this.landCache.asMap().entrySet().stream().filter(entry -> ((Land)entry.getValue()).isCoOwner(playerId)).map(Map.Entry::getKey).findFirst().orElseThrow(() -> new IllegalStateException("Not a co-owner of any land"));
        return this.removeCoOwner(ownerId, playerId).thenApply(v -> null);
    }

    public Collection<Land> getAllLands() {
        return Collections.unmodifiableCollection(this.landCache.asMap().values());
    }

    public int calculateLandSize(int level) {
        int sizeIncrease = (level - 1) * this.sizeIncreasePerLevel;
        return Math.min(this.initialSize + sizeIncrease, this.maxFarmSize);
    }

    private Cuboid createCuboidForFarm(@NotNull Location location, int size) {
        double halfSize = (double)size / 2.0;
        Location corner1 = location.clone();
        corner1.add(-halfSize, 0.0, -halfSize);
        corner1.setY((double)this.depthLimit);
        Location corner2 = location.clone();
        corner2.add(halfSize, 0.0, halfSize);
        corner2.setY((double)this.heightLimit);
        return Cuboid.from(corner1, corner2);
    }

    private Location calculateDefaultHome(@NotNull Cuboid cuboid) {
        Location center = cuboid.getCenter();
        center.setY((double)this.heightLimit);
        for (int y = this.heightLimit; y >= this.depthLimit; --y) {
            Location check = new Location(center.getWorld(), center.getX(), (double)y, center.getZ());
            if (check.getBlock().getType().isAir()) continue;
            return new Location(center.getWorld(), center.getX(), (double)(y + 1), center.getZ());
        }
        return new Location(center.getWorld(), center.getX(), (double)(this.depthLimit + 1), center.getZ());
    }

    public boolean hasPendingInvitation(@NotNull UUID playerId) {
        return this.pendingInvitations.values().stream().anyMatch(invites -> invites.contains(playerId));
    }

    public void invalidateCache(@NotNull UUID ownerId) {
        this.landCache.invalidate(ownerId);
    }

    public void clearCache() {
        this.landCache.invalidateAll();
        this.coOwnerToOwnerCache.invalidateAll();
    }
}

