/*
 * Decompiled with CFR 0.152.
 */
package fr.skynex.mycommands.managers;

import fr.skynex.mycommands.MyCommands;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Location;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitTask;

public class CacheManager {
    private final MyCommands plugin;
    private final Map<UUID, Map<String, CachedLocation>> homesCache = new ConcurrentHashMap<UUID, Map<String, CachedLocation>>();
    private final Map<String, CachedLocation> warpsCache = new ConcurrentHashMap<String, CachedLocation>();
    private CachedLocation spawnCache = null;
    private long cacheExpiration;
    private boolean cacheEnabled;
    private BukkitTask cleanupTask;
    private long cacheHits = 0L;
    private long cacheMisses = 0L;

    public CacheManager(MyCommands plugin) {
        this.plugin = plugin;
        this.loadConfig();
        this.startCleanupTask();
    }

    private void loadConfig() {
        this.cacheEnabled = this.plugin.getConfig().getBoolean("cache.enabled", true);
        this.cacheExpiration = this.plugin.getConfig().getLong("cache.expiration-seconds", 300L) * 1000L;
    }

    private void startCleanupTask() {
        if (!this.cacheEnabled) {
            return;
        }
        this.cleanupTask = this.plugin.getServer().getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, this::cleanupExpiredEntries, 1200L, 1200L);
    }

    public Location getHome(UUID uuid, String homeName) {
        if (!this.cacheEnabled) {
            return null;
        }
        Map<String, CachedLocation> playerHomes = this.homesCache.get(uuid);
        if (playerHomes == null) {
            ++this.cacheMisses;
            return null;
        }
        CachedLocation cached = playerHomes.get(homeName);
        if (cached == null || cached.isExpired()) {
            ++this.cacheMisses;
            if (cached != null) {
                playerHomes.remove(homeName);
            }
            return null;
        }
        ++this.cacheHits;
        return cached.getLocation();
    }

    public void putHome(UUID uuid, String homeName, Location location) {
        if (!this.cacheEnabled || location == null) {
            return;
        }
        this.homesCache.computeIfAbsent(uuid, k -> new ConcurrentHashMap()).put(homeName, new CachedLocation(location, this.cacheExpiration));
    }

    public void removeHome(UUID uuid, String homeName) {
        Map<String, CachedLocation> playerHomes = this.homesCache.get(uuid);
        if (playerHomes != null) {
            playerHomes.remove(homeName);
        }
    }

    public Map<String, Location> getAllHomes(UUID uuid) {
        if (!this.cacheEnabled) {
            return null;
        }
        Map<String, CachedLocation> playerHomes = this.homesCache.get(uuid);
        if (playerHomes == null || playerHomes.isEmpty()) {
            ++this.cacheMisses;
            return null;
        }
        HashMap<String, Location> result = new HashMap<String, Location>();
        for (Map.Entry<String, CachedLocation> entry : playerHomes.entrySet()) {
            if (entry.getValue().isExpired()) continue;
            result.put(entry.getKey(), entry.getValue().getLocation());
        }
        if (result.isEmpty()) {
            ++this.cacheMisses;
            return null;
        }
        ++this.cacheHits;
        return result;
    }

    public void invalidatePlayerHomes(UUID uuid) {
        this.homesCache.remove(uuid);
    }

    public Location getWarp(String warpName) {
        if (!this.cacheEnabled) {
            return null;
        }
        CachedLocation cached = this.warpsCache.get(warpName);
        if (cached == null || cached.isExpired()) {
            ++this.cacheMisses;
            if (cached != null) {
                this.warpsCache.remove(warpName);
            }
            return null;
        }
        ++this.cacheHits;
        return cached.getLocation();
    }

    public void putWarp(String warpName, Location location) {
        if (!this.cacheEnabled || location == null) {
            return;
        }
        this.warpsCache.put(warpName, new CachedLocation(location, this.cacheExpiration));
    }

    public void removeWarp(String warpName) {
        this.warpsCache.remove(warpName);
    }

    public Map<String, Location> getAllWarps() {
        if (!this.cacheEnabled || this.warpsCache.isEmpty()) {
            ++this.cacheMisses;
            return null;
        }
        HashMap<String, Location> result = new HashMap<String, Location>();
        for (Map.Entry<String, CachedLocation> entry : this.warpsCache.entrySet()) {
            if (entry.getValue().isExpired()) continue;
            result.put(entry.getKey(), entry.getValue().getLocation());
        }
        if (result.isEmpty()) {
            ++this.cacheMisses;
            return null;
        }
        ++this.cacheHits;
        return result;
    }

    public void invalidateAllWarps() {
        this.warpsCache.clear();
    }

    public Location getSpawn() {
        if (!this.cacheEnabled || this.spawnCache == null || this.spawnCache.isExpired()) {
            ++this.cacheMisses;
            return null;
        }
        ++this.cacheHits;
        return this.spawnCache.getLocation();
    }

    public void putSpawn(Location location) {
        if (!this.cacheEnabled || location == null) {
            return;
        }
        this.spawnCache = new CachedLocation(location, this.cacheExpiration);
    }

    public void invalidateSpawn() {
        this.spawnCache = null;
    }

    private void cleanupExpiredEntries() {
        int removed = 0;
        for (Map<String, CachedLocation> playerHomes : this.homesCache.values()) {
            int beforeSize = playerHomes.size();
            playerHomes.entrySet().removeIf(entry -> ((CachedLocation)entry.getValue()).isExpired());
            removed += beforeSize - playerHomes.size();
        }
        this.homesCache.entrySet().removeIf(entry -> ((Map)entry.getValue()).isEmpty());
        int beforeSize = this.warpsCache.size();
        this.warpsCache.entrySet().removeIf(entry -> ((CachedLocation)entry.getValue()).isExpired());
        removed += beforeSize - this.warpsCache.size();
        if (this.spawnCache != null && this.spawnCache.isExpired()) {
            this.spawnCache = null;
            ++removed;
        }
        if (removed > 0 && this.plugin.getConfig().getBoolean("debug.enabled", false)) {
            this.plugin.getLogger().info(String.format("Cache cleanup: %d entr\u00c3\u00a9es expir\u00c3\u00a9es supprim\u00c3\u00a9es", removed));
        }
    }

    public void clearAll() {
        this.homesCache.clear();
        this.warpsCache.clear();
        this.spawnCache = null;
        this.plugin.getLogger().info("Cache vid\u00c3\u00a9 compl\u00c3\u00a8tement");
    }

    public void shutdown() {
        if (this.cleanupTask != null) {
            this.cleanupTask.cancel();
        }
        this.clearAll();
    }

    public void reload() {
        this.loadConfig();
        this.clearAll();
    }

    public Map<String, Object> getStats() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        long total = this.cacheHits + this.cacheMisses;
        double hitRate = total > 0L ? (double)this.cacheHits / (double)total * 100.0 : 0.0;
        stats.put("enabled", this.cacheEnabled);
        stats.put("hits", this.cacheHits);
        stats.put("misses", this.cacheMisses);
        stats.put("total", total);
        stats.put("hit_rate", String.format("%.2f%%", hitRate));
        stats.put("homes_cached", this.homesCache.values().stream().mapToInt(Map::size).sum());
        stats.put("warps_cached", this.warpsCache.size());
        stats.put("spawn_cached", this.spawnCache != null);
        return stats;
    }

    public void resetStats() {
        this.cacheHits = 0L;
        this.cacheMisses = 0L;
    }

    private static class CachedLocation {
        private final Location location;
        private final long expiresAt;

        public CachedLocation(Location location, long ttl) {
            this.location = location.clone();
            this.expiresAt = System.currentTimeMillis() + ttl;
        }

        public Location getLocation() {
            return this.location.clone();
        }

        public boolean isExpired() {
            return System.currentTimeMillis() > this.expiresAt;
        }
    }
}

